/*:
 * @target MZ
 * @plugindesc v1.0 Link HS_LangSimple with HS_OptionHelp: per-language fallbacks + in-game variable override
 * @author HS
 *
 * @help
 * タイトルで HS_LangSimple の言語を切り替えた際、HS_OptionHelp のヘルプ文も
 * 自動で多言語切替します。ゲーム中は「変数の文字列」を優先し、未設定時は
 * 言語別フォールバック（本プラグインのパラメータ）を使用します。
 *
 * 【順序】HS_LangSimple →（任意のLangブリッジ）→ HS_OptionHelp → 本プラグイン
 *
 * 【対応シンボル】fastMode / hsAuto / hsProtect
 *  ※他シンボルは元の HS_OptionHelp の動作にフォールバックします。
 *
 * @param fastFb0 @text 高速:0 日本語 @default テキストや演出の待機を短縮します。
 * @param fastFb1 @text 高速:1 English @default Shortens wait times for text and effects.
 * @param fastFb2 @text 高速:2 简体中文 @default 缩短文本与演出等待时间。
 * @param fastFb3 @text 高速:3 한국어 @default 텍스트와 연출의 대기 시간을 단축합니다.
 *
 * @param autoFb0 @text 自動:0 日本語 @default メッセージを一定時間で自動で進めます。
 * @param autoFb1 @text 自動:1 English @default Automatically advances messages at set intervals.
 * @param autoFb2 @text 自動:2 简体中文 @default 以设定时间间隔自动推进文本。
 * @param autoFb3 @text 自動:3 한국어 @default 메시지를 설정된 간격으로 자동 진행합니다.
 *
 * @param protFb0 @text 保護:0 日本語 @default ONにすると重要イベント中は高速/自動送りを停止します。OFFにすると重要イベントも流れます。
 * @param protFb1 @text 保護:1 English @default When ON, Fast/Auto is disabled during important events. When OFF, important events will also advance.
 * @param protFb2 @text 保護:2 简体中文 @default 开启时，重要事件期间将停用“高速/自动推进”。关闭时，重要事件也会照常推进。
 * @param protFb3 @text 保護:3 한국어 @default ON이면 중요한 이벤트 중에는 고속/자동 진행이 중지됩니다. OFF이면 중요한 이벤트도 진행됩니다.
 */
(() => {
  "use strict";
  const PN = "HS_OptionHelp_LangBridge";
  const P  = PluginManager.parameters(PN);

  // 言語indexは HS_LangSimple が ConfigManager.hs_langIndex に保持
  const langIndex = () => Number(ConfigManager && ConfigManager.hs_langIndex != null
    ? ConfigManager.hs_langIndex : 0);

  // HS_OptionHelp の「変数ID」を参照して、ゲーム中はその“文字列”を優先
  const prmOH = PluginManager.parameters("HS_OptionHelp") || {};
  const VAR_IDS = {
    fastMode : Number(prmOH["fastModeVar"]   || 0),
    hsAuto   : Number(prmOH["hsAutoVar"]     || 0),
    hsProtect: Number(prmOH["hsProtectVar"]  || 0),
  };
  const isNonEmptyString = v => typeof v === "string" && v.trim() !== "";
  function varString(id){
    if (id > 0 && window.$gameVariables && $gameVariables._data){
      const v = $gameVariables.value(id);
      if (isNonEmptyString(v)) return String(v);
    }
    return null;
  }

  // 言語別フォールバック（タイトルや未初期化時はこちらを使用）
  const FB = {
    fastMode:  [P.fastFb0||"", P.fastFb1||"", P.fastFb2||"", P.fastFb3||""],
    hsAuto:    [P.autoFb0||"", P.autoFb1||"", P.autoFb2||"", P.autoFb3||""],
    hsProtect: [P.protFb0||"", P.protFb1||"", P.protFb2||"", P.protFb3||""],
  };
  function fbText(symbol){
    const arr = FB[symbol];
    if (!arr) return "";
    const i = langIndex();
    return arr[i] || arr[0] || "";
  }

  // 既存の updateHsOptionHelp を保持
  const _update = Scene_Options.prototype.updateHsOptionHelp;

  // 対象シンボルだけ差し替え（その他は元処理）
  Scene_Options.prototype.updateHsOptionHelp = function(){
    // 既存ウィンドウがなければ元処理
    if (!this._hsOptionHelpWindow || !this._optionsWindow){
      return _update ? _update.call(this) : undefined;
    }
    const sym = this._optionsWindow.commandSymbol(this._optionsWindow.index());
    if (sym === "fastMode" || sym === "hsAuto" || sym === "hsProtect"){
      const v = varString(VAR_IDS[sym]);           // ゲーム中：変数の文字列を優先
      const t = isNonEmptyString(v) ? v : fbText(sym); // それ以外：言語別FB
      this._hsOptionHelpWindow.setText(t);
      return; // 他は触らない
    }
    // 対象外は元の挙動で
    if (_update) return _update.call(this);
  };
})();
